/**
 * VAPG Player - 授权验证解密器（胶水代码库）
 * 
 * 版权所有 (c) 2024 北京锋通科技有限公司
 * 
 * 本软件受版权法和国际条约保护。未经北京锋通科技有限公司明确书面许可，
 * 不得以任何形式或任何方式（电子、机械、影印、录制或其他方式）复制、
 * 传播、修改、分发或使用本软件的任何部分。
 * 
 * 许可信息：
 * 本软件仅供授权用户使用，禁止未经授权的复制、分发或修改。
 * 
 * 免责声明：
 * 本软件按"原样"提供，不提供任何明示或暗示的担保，包括但不限于
 * 对适销性、特定用途适用性和非侵权性的担保。
 * 
 * 技术支持：
 * 北京锋通科技有限公司
 * 官方网站: https://www.ftpeak.com
 * 
 * 版本: 1.0.0
 * 发布日期: 2024年
 */

(function(global) {
    'use strict';
    
    /**
     * 动态计算WASM文件路径
     * @returns {string} WASM文件所在目录的绝对路径
     */
    function getWasmPath() {
        // 基于当前脚本位置计算路径
        var scripts = document.getElementsByTagName('script');
        var currentScript = scripts[scripts.length - 1];
        var scriptPath = currentScript.src;
        
        // 提取脚本所在目录
        var scriptDir = scriptPath.substring(0, scriptPath.lastIndexOf('/') + 1);
        
        return scriptDir;
    }
    
    // Worker脚本内容 - 使用特殊占位符避免JavaScript解释
    var workerScript = `
        let wasmModule = null;
        let wasmReady = false;
        
        /**
         * 初始化WASM模块
         */
        async function initWasm() {
            try {
                // 使用动态计算的路径加载WASM JS绑定文件
                importScripts('__WASM_JS_PATH__');
                
                // 初始化WASM模块 - 使用动态计算的路径
                await wasm_bindgen({ module_or_path: '__WASM_WASM_PATH__' });
                
                // 从wasm_bindgen对象获取函数
                wasmModule = {
                    process_data: wasm_bindgen.process_data,
                    process_string: wasm_bindgen.process_string
                };
                wasmReady = true;
            } catch (error) {
                console.error('VAPG: 模块加载失败:', error);
                wasmReady = false;
            }
        }
        
        // 初始化WASM
        initWasm();
        
        /**
         * Worker消息处理函数
         * @param {MessageEvent} e - 消息事件对象
         */
        self.onmessage = async function(e) {
            var data = e.data;
            var id = data.id;
            var url = data.url;
            var type = data.type;
            var fragmentInfo = data.fragmentInfo;
            
            try {
                var processedData = data.data;
                var workerProcessed = true;
                
                // 等待WASM模块加载完成
                if (!wasmReady) {
                    var waitStart = Date.now();
                    while (!wasmReady && Date.now() - waitStart < 5000) {
                        await new Promise(resolve => setTimeout(resolve, 100));
                    }
                    if (!wasmReady) {
                        workerProcessed = false;
                    }
                }
                
                // 如果WASM模块已就绪，则处理数据
                if (wasmReady) {
                    // 根据数据类型调用相应的处理函数
                    if (type === 'decrypt-ts' || type === 'arraybuffer') {
                        processedData = await decryptData(data.data, fragmentInfo, url);
                    } else if (type === 'process-manifest' || typeof data.data === 'string') {
                        processedData = await processStringData(data.data, url);
                    }
                }
                
                self.postMessage({
                    id: id,
                    data: processedData,
                    url: url,
                    workerProcessed: workerProcessed,
                    type: type
                });
            } catch (error) {
                self.postMessage({
                    id: id,
                    data: data.data,
                    url: url,
                    workerProcessed: false,
                    error: error.message,
                    type: type
                });
            }
        };

        /**
         * 解密数据函数
         * @param {ArrayBuffer} encryptedData - 加密的数据
         * @param {Object} fragmentInfo - 片段信息
         * @param {string} url - 数据URL
         * @returns {ArrayBuffer} 解密后的数据
         */
        async function decryptData(encryptedData, fragmentInfo, url) {
            // 强制等待WASM模块加载完成
            if (!wasmReady) {
                let waitTime = 0;
                while (!wasmReady && waitTime < 5000) {
                    await new Promise(resolve => setTimeout(resolve, 100));
                    waitTime += 100;
                }
            }
            
            // 如果WASM模块已准备好，通过WASM处理数据
            if (wasmReady && wasmModule) {
                // 将ArrayBuffer转换为Uint8Array
                const dataArray = new Uint8Array(encryptedData);
                
                // 检查url是否存在，如果不存在则赋默认值
                const safeUrl = url || "";
                
                // 通过WASM处理数据
                const processedArray = wasmModule.process_data(dataArray, safeUrl);
                
                // 返回处理后的ArrayBuffer
                return processedArray.buffer;
            } else {
                return encryptedData;
            }
        }
        
        /**
         * 处理字符串数据函数
         * @param {string} stringData - 字符串数据
         * @param {string} url - 数据URL
         * @returns {string} 处理后的字符串
         */
        async function processStringData(stringData, url) {
            // 如果WASM模块已准备好，通过WASM处理字符串数据
            if (wasmReady && wasmModule) {
                // 检查url是否存在，如果不存在则赋默认值
                const safeUrl = url || "";
                
                // 通过WASM处理字符串
                const processedString = wasmModule.process_string(stringData, safeUrl);
                
                // 添加处理标记
                return processedString + "\\n# ProcessedByWASM_" + Date.now();
            } else {
                return stringData + "\\n# ProcessedByWorker_" + Date.now();
            }
        }
    `;
    
    /**
     * Worker管理器 - 单例模式
     * 负责创建和管理Web Worker，处理数据加解密任务
     */
    var WorkerManager = (function() {
        var instance = null;
        
        function WorkerManager() {
            if (instance) {
                return instance;
            }
            instance = this;
            
            this.worker = null;
            this.initialized = false;
            this.requestId = 0;
            this.pendingRequests = new Map();
            this.initPromise = null;
            
            return this;
        }
        
        /**
         * 初始化Worker
         * @returns {Promise} 初始化完成的Promise
         */
        WorkerManager.prototype.init = function() {
            var self = this;
            
            if (this.initPromise) {
                return this.initPromise;
            }
            
            this.initPromise = new Promise(function(resolve, reject) {
                try {
                    // 动态计算WASM文件路径
                    var wasmBasePath = getWasmPath();
                    var wasmJsPath = wasmBasePath + 'hls_wasm.js';
                    var wasmWasmPath = wasmBasePath + 'hls_wasm_bg.wasm';
                    
                    // 替换Worker脚本中的路径占位符
                    var processedWorkerScript = workerScript
                        .replace(/__WASM_JS_PATH__/g, wasmJsPath)
                        .replace(/__WASM_WASM_PATH__/g, wasmWasmPath);
                    
                    var workerBlob = new Blob([processedWorkerScript], { type: 'application/javascript' });
                    self.worker = new Worker(URL.createObjectURL(workerBlob));
                    
                    self.worker.onmessage = function(e) {
                        var data = e.data;
                        var id = data.id;
                        var workerProcessed = data.workerProcessed;
                        var error = data.error;
                        
                        if (self.pendingRequests.has(id)) {
                            var request = self.pendingRequests.get(id);
                            var resolveFunc = request.resolve;
                            var rejectFunc = request.reject;
                            self.pendingRequests.delete(id);
                            
                            if (error) {
                                rejectFunc(new Error(error));
                            } else {
                                resolveFunc({ data: data.data, workerProcessed: workerProcessed });
                            }
                        }
                    };
                    
                    self.worker.onerror = function(error) {
                        console.error('Worker错误:', error);
                        reject(new Error('Worker运行错误: ' + error.message));
                    };
                    
                    var testId = self.requestId++;
                    self.pendingRequests.set(testId, { 
                        resolve: function() {
                            self.initialized = true;
                            resolve();
                        },
                        reject: reject
                    });
                    
                    self.worker.postMessage({ 
                        id: testId, 
                        data: 'test', 
                        url: 'test://worker_init',
                        type: 'string',
                        fragmentInfo: null
                    });
                    
                } catch (error) {
                    reject(new Error('创建Worker失败: ' + error.message));
                }
            });
            
            return this.initPromise;
        };
        
        /**
         * 处理数据
         * @param {ArrayBuffer|string} data - 要处理的数据
         * @param {string} url - 数据URL
         * @param {string} type - 数据类型
         * @param {Object} fragmentInfo - 片段信息
         * @returns {Promise} 处理结果的Promise
         */
        WorkerManager.prototype.processData = function(data, url, type, fragmentInfo) {
            var self = this;
            
            if (type === undefined) type = 'string';
            if (fragmentInfo === undefined) fragmentInfo = null;
            
            return new Promise(function(resolve, reject) {
                var initWorker = function() {
                    if (!self.initialized) {
                        if (self.initPromise) {
                            return self.initPromise;
                        } else {
                            return self.init();
                        }
                    }
                    return Promise.resolve();
                };
                
                initWorker().then(function() {
                    var id = self.requestId++;
                    self.pendingRequests.set(id, { resolve: resolve, reject: reject });
                    
                    var messageType = type === 'arraybuffer' ? 'decrypt-ts' : 'process-manifest';
                    var message = {
                        id: id, 
                        type: messageType,
                        data: data, 
                        url: url,
                        fragmentInfo: self.sanitizeFragmentInfo(fragmentInfo)
                    };
                    
                    if (type === 'arraybuffer' && data instanceof ArrayBuffer) {
                        self.worker.postMessage(message, [data]);
                    } else {
                        self.worker.postMessage(message);
                    }
                }).catch(reject);
            });
        };
        
        /**
         * 清理片段信息
         * @param {Object} fragmentInfo - 原始片段信息
         * @returns {Object} 清理后的片段信息
         */
        WorkerManager.prototype.sanitizeFragmentInfo = function(fragmentInfo) {
            if (!fragmentInfo) return null;
            
            return {
                url: fragmentInfo.url,
                sn: fragmentInfo.sn,
                level: fragmentInfo.level,
                type: fragmentInfo.type,
                baseUrl: fragmentInfo.baseUrl,
                cc: fragmentInfo.cc,
                start: fragmentInfo.start,
                duration: fragmentInfo.duration
            };
        };
        
        /**
         * 销毁Worker管理器
         */
        WorkerManager.prototype.destroy = function() {
            if (this.worker) {
                this.worker.terminate();
                this.worker = null;
            }
            this.pendingRequests.clear();
            this.initialized = false;
            this.initPromise = null;
            instance = null;
        };
        
        return WorkerManager;
    })();
    
    /**
     * 自定义片段加载器 - 继承自HLS默认加载器
     * 负责拦截和处理HLS流数据
     */
    var TrackingFragmentLoader = function(config) {
        // 调用父类构造函数
        Hls.DefaultConfig.loader.call(this, config);
        this.workerManager = globalWorkerManager;
    };

    // 设置原型链继承
    TrackingFragmentLoader.prototype = Object.create(Hls.DefaultConfig.loader.prototype);
    TrackingFragmentLoader.prototype.constructor = TrackingFragmentLoader;

    /**
     * 加载方法重写
     * @param {Object} context - 加载上下文
     * @param {Object} config - 配置对象
     * @param {Object} callbacks - 回调函数对象
     */
    TrackingFragmentLoader.prototype.load = function(context, config, callbacks) {
        var self = this;
        var fileName = context.url.split('/').pop() || context.url;
        
        var originalOnSuccess = callbacks.onSuccess;
        
        callbacks.onSuccess = function(response, stats, context) {
            var fileName = context.url.split('/').pop() || context.url;
            var isPlaylist = context.type === 'manifest' || fileName.endsWith('.m3u8');
            
            var processData = function() {
                return new Promise(function(resolve, reject) {
                    try {
                        var dataToProcess;
                        var dataType = 'string';
                        
                        if (isPlaylist) {
                            dataToProcess = typeof response.data === 'string' 
                                ? response.data 
                                : new TextDecoder().decode(response.data);
                        } else {
                            dataToProcess = response.data;
                            dataType = 'arraybuffer';
                        }
                        
                        var frag = context.frag || null;
                        
                        self.workerManager.processData(
                            dataToProcess, 
                            context.url, 
                            dataType,
                            frag
                        ).then(function(result) {
                            var processedResponse = {};
                            for (var key in response) {
                                if (response.hasOwnProperty(key)) {
                                    processedResponse[key] = response[key];
                                }
                            }
                            
                            if (dataType === 'arraybuffer') {
                                // 媒体文件：保持ArrayBuffer格式
                                processedResponse.data = result.data;
                            } else {
                                // 播放列表文件：保持字符串格式
                                processedResponse.data = result.data;
                            }
                            
                            resolve(processedResponse);
                        }).catch(function(error) {
                            reject(error);
                        });
                    } catch (error) {
                        reject(error);
                    }
                });
            };
            
            processData().then(function(processedResponse) {
                originalOnSuccess.call(self, processedResponse, stats, context);
            }).catch(function(error) {
                addLog('VAPG：处理失败: ' + fileName + ' - ' + error.message);
                originalOnSuccess.call(self, response, stats, context);
            });
        };
        
        // 调用父类load方法
        Hls.DefaultConfig.loader.prototype.load.call(this, context, config, callbacks);
    };

    /**
     * 销毁方法重写
     */
    TrackingFragmentLoader.prototype.destroy = function() {
        Hls.DefaultConfig.loader.prototype.destroy.call(this);
    };
    
    // 全局单例Worker管理器
    var globalWorkerManager = new WorkerManager();
    
    /**
     * 日志函数
     * @param {string} message - 日志消息
     */
    function addLog(message) {
        var log = document.getElementById('log');
        if (log) {
            var entry = document.createElement('div');
            entry.textContent = '[' + new Date().toLocaleTimeString() + '] ' + message;
            log.appendChild(entry);
            log.scrollTop = log.scrollHeight;
        }
        console.log(message);
    }
    
    /**
     * HLS播放器初始化函数
     * @param {Object} options - 配置选项
     * @returns {Object} HLS播放器实例
     */
    function initHLSPlayer(options) {
        var defaults = {
            videoElement: null,
            videojsPlayer: null,
            videoUrl: null,
            logContainer: 'log',
            debug: false
        };
        
        var config = Object.assign({}, defaults, options);
        
        // 检测传入的是Video.js实例还是原生video元素
        var videoElement;
        if (config.videojsPlayer) {
            // Video.js模式：从Video.js实例获取原生video元素
            videoElement = config.videojsPlayer.el().querySelector('video');
            if (!videoElement) {
                throw new Error('无法从Video.js实例获取视频元素');
            }
        } else if (config.videoElement) {
            // 原生模式：直接使用传入的video元素
            videoElement = config.videoElement;
        } else {
            throw new Error('请提供videoElement或videojsPlayer参数');
        }
        
        if (!config.videoUrl) {
            throw new Error('请配置视频URL');
        }
        
        var hls;

        if (Hls.isSupported()) {
            // 一次性Worker初始化（在创建HLS实例之前）
            globalWorkerManager.init()
                .then(function() {
                    // 启用播放按钮（如果存在）
                    var playBtn = document.getElementById('playBtn');
                    if (playBtn) {
                        playBtn.disabled = false;
                        playBtn.textContent = '开始播放';
                    }
                    
                    // Worker初始化成功后创建HLS实例
                    hls = new Hls({
                        // 片段加载器
                        fLoader: function(config) {
                            return new TrackingFragmentLoader(config);
                        },
                        // 播放列表加载器
                        pLoader: function(config) {
                            return new TrackingFragmentLoader(config);
                        },
                        debug: config.debug,
                        enableWorker: false,
                        enableSoftwareAES: true,
                        lowLatencyMode: true,
                        backBufferLength: 90
                    });

                    hls.attachMedia(videoElement);

                    hls.on(Hls.Events.ERROR, function(event, data) {
                        if (data.fatal) {
                            addLog('致命错误: ' + data.details);
                            switch (data.type) {
                                case Hls.ErrorTypes.NETWORK_ERROR:
                                    addLog('网络错误，尝试恢复...');
                                    hls.startLoad();
                                    break;
                                case Hls.ErrorTypes.MEDIA_ERROR:
                                    addLog('媒体错误，尝试恢复...');
                                    hls.recoverMediaError();
                                    break;
                                default:
                                    addLog('无法恢复的错误，重新初始化...');
                                    hls.destroy();
                                    break;
                            }
                        } else {
                            addLog('非致命错误: ' + data.details);
                        }
                    });

                    // 播放按钮点击事件
                    if (playBtn) {
                        playBtn.addEventListener('click', function() {
                            if (hls) {
                                videoElement.play().catch(function(err) {
                                    console.error('VAPG：视频播放错误:', err);
                                });
                                playBtn.disabled = true;
                                playBtn.textContent = '播放中...';
                            }
                        });
                    }

                    // 初始加载媒体
                    hls.loadSource(config.videoUrl);
                })
                .catch(function(error) {
                    addLog('✗VAPG：初始化失败: ' + error.message);
                    var playBtn = document.getElementById('playBtn');
                    if (playBtn) {
                        playBtn.disabled = true;
                        playBtn.textContent = 'Worker初始化失败';
                    }
                });

        } else if (videoElement.canPlayType('application/vnd.apple.mpegurl')) {
            videoElement.src = config.videoUrl;
            
            var playBtn = document.getElementById('playBtn');
            if (playBtn) {
                playBtn.disabled = false;
                playBtn.textContent = '开始播放';
                
                playBtn.addEventListener('click', function() {
                    videoElement.play().catch(function(err) {
                        addLog('播放失败: ' + err.message);
                    });
                    playBtn.disabled = true;
                    playBtn.textContent = '播放中...';
                });
            }
        }

        // 页面卸载时清理
        window.addEventListener('beforeunload', function() {
            if (hls) {
                hls.destroy();
            }
            if (globalWorkerManager) {
                globalWorkerManager.destroy();
            }
        });
        
        return {
            hls: hls,
            destroy: function() {
                if (hls) {
                    hls.destroy();
                }
                if (globalWorkerManager) {
                    globalWorkerManager.destroy();
                }
            }
        };
    }
    
    // 自动初始化（如果页面有相应元素）
    function autoInit() {
        if (document.getElementById('video')) {
            // 清空日志按钮事件
            var clearBtn = document.getElementById('clearLog');
            if (clearBtn) {
                clearBtn.addEventListener('click', function() {
                    var log = document.getElementById('log');
                    if (log) {
                        log.innerHTML = '';
                    }
                });
            }
            
            // 检查是否有预配置的视频URL
            var videoElement = document.getElementById('video');
            var videoUrl = videoElement.getAttribute('data-video-url');
            
            if (videoUrl) {
                // 如果有预配置的URL，则自动初始化
                return initHLSPlayer({
                    videoElement: videoElement,
                    videoUrl: videoUrl
                });
            } else {
                // 如果没有预配置URL，则等待手动初始化
                return null;
            }
        }
        return null;
    }
    
    // 暴露全局API
    global.HLSPlayerLegacy = {
        init: initHLSPlayer,
        autoInit: autoInit,
        addLog: addLog,
        WorkerManager: WorkerManager,
        TrackingFragmentLoader: TrackingFragmentLoader
    };
    
    // 页面加载完成后自动初始化
    document.addEventListener('DOMContentLoaded', function() {
        global.HLSPlayerLegacy.autoInit();
    });
    
})(window);

/**
 * 文件结束 - VAPG Player
 * 
 * 版权所有 (c) 2024 北京锋通科技有限公司
 * 保留所有权利
 * 
 * 版本: 1.0.0
 * 文件: vapg_player.js
 * 描述: 授权验证解密器的胶水代码库，支持授权校验、解密和Web Worker处理
 */